async function downloadImage(url, filename) {
    try {
        const response = await fetch(url);
        const blob = await response.blob();
        const blobUrl = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = blobUrl;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(blobUrl);
        a.remove();
    } catch (e) { window.open(url, '_blank'); }
}

async function copyImageToClipboard(url) {
    try {
        const response = await fetch(url);
        const blob = await response.blob();
        const item = new ClipboardItem({ 'image/png': new Blob([blob], { type: 'image/png' }) });
        await navigator.clipboard.write([item]);
        alert("Copied to clipboard!");
    } catch (err) { alert("Copy failed."); }
}

chrome.runtime.onMessage.addListener((request) => {
    if (request.action === "download") downloadImage(request.url, `thumb-${request.id}.jpg`);
    if (request.action === "copy") copyImageToClipboard(request.url);
});

function injectThumbnail() {
    const videoId = new URLSearchParams(window.location.search).get("v");
    if (!videoId) return;

    const secondaryInner = document.querySelector("#secondary-inner") || document.querySelector("#secondary");
    if (!secondaryInner) {
        setTimeout(injectThumbnail, 1000);
        return;
    }

    let container = document.getElementById("thumb-ext-container");
    const imgUrl = `https://img.youtube.com/vi/${videoId}/maxresdefault.jpg`;

    if (!container) {
        container = document.createElement("div");
        container.id = "thumb-ext-container";
        container.innerHTML = `
            <img src="${imgUrl}" id="thumb-ext-preview" onerror="this.src='https://img.youtube.com/vi/${videoId}/hqdefault.jpg'">
            <div class="thumb-btn-group">
                <button class="thumb-action-btn copy-btn">Copy</button>
                <button class="thumb-action-btn dl-btn">Download</button>
            </div>
        `;
        
        let modal = document.getElementById("thumb-modal") || document.createElement("div");
        modal.id = "thumb-modal";
        if (!modal.parentElement) document.body.appendChild(modal);
        modal.innerHTML = `
            <img src="${imgUrl}">
            <div class="modal-button-container">
                <button class="modal-action-btn modal-copy">Copy to Clipboard</button>
                <button class="modal-action-btn modal-dl">Download Full Res</button>
            </div>
            <p style="color:#aaa; margin-top:20px; font-family:sans-serif; font-size:12px;">Click background to close</p>
        `;

        modal.querySelector(".modal-dl").onclick = (e) => { e.stopPropagation(); downloadImage(imgUrl, `thumb-${videoId}.jpg`); };
        modal.querySelector(".modal-copy").onclick = (e) => { e.stopPropagation(); copyImageToClipboard(imgUrl); };
        modal.onclick = (e) => { if(e.target === modal || e.target.tagName === 'P') modal.style.display = "none"; };

        container.querySelector(".dl-btn").onclick = (e) => { e.stopPropagation(); downloadImage(imgUrl, `thumb-${videoId}.jpg`); };
        container.querySelector(".copy-btn").onclick = (e) => { e.stopPropagation(); copyImageToClipboard(imgUrl); };
        container.onclick = (e) => { if(!e.target.classList.contains('thumb-action-btn')) modal.style.display = "flex"; };
    }

    if (secondaryInner.firstChild !== container) {
        secondaryInner.prepend(container);
    }
}

let lastUrl = location.href;
const observer = new MutationObserver(() => {
    if (location.href !== lastUrl) {
        lastUrl = location.href;
        document.getElementById("thumb-ext-container")?.remove();
        injectThumbnail();
    } else {
        injectThumbnail();
    }
});

observer.observe(document.body, { subtree: true, childList: true });
injectThumbnail();